/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private define ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
ADC_HandleTypeDef        hADC;
DMA_HandleTypeDef        hDMA;
uint32_t                 uhADCxConvertedData[4];

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void APP_AdcConfig(void);

/**
  * @brief  Main program.
  * @retval int
  */
int main(void)
{
  /* Reset of all peripherals, Initializes the Systick. */
  HAL_Init();

  /* System clock configuration */
  APP_SystemClockConfig(); 

  /* Initialize UART */
  BSP_UART_Config();

  /* Configure ADC */
  APP_AdcConfig();
   
  /* ADC Calibrate */   
  if (HAL_ADCEx_Calibration_Start(&hADC, ADC_CALIBRATION_NUMBER_1) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  
  /* Infinite loop */
  while (1)
  {
    HAL_ADC_Start_DMA(&hADC, (uint32_t*)uhADCxConvertedData, 4);
    BSP_LED_Toggle(LED_GREEN);
    while(HAL_DMA_GetState(hADC.DMA_Handle) != HAL_DMA_STATE_READY);
    
    HAL_Delay(1000);
  }
}

/**
  * @brief  Conversion complete callback in non-blocking mode.
  * @param  hadc ADC handle
  * @retval None
  */
void HAL_ADC_ConvCpltCallback(ADC_HandleTypeDef *hadc)
{
  printf("Channel4: %u \r\n",(unsigned int)uhADCxConvertedData[0]);
  printf("Channel5: %u \r\n",(unsigned int)uhADCxConvertedData[1]);
  printf("Channel6: %u \r\n",(unsigned int)uhADCxConvertedData[2]);
  printf("Channel7: %u \r\n",(unsigned int)uhADCxConvertedData[3]);
}

/**
  * @brief  ADC Configuration
  * @param  None
  * @retval None
  */
static void APP_AdcConfig(void)
{
  ADC_ChannelConfTypeDef   sConfig = {0};
  
  hADC.Instance = ADC1;
  hADC.Init.ClockPrescaler        = ADC_CLOCK_SYNC_PCLK_DIV4;         /* ADC sync clock with prescaler division by 4 */
  hADC.Init.Resolution            = ADC_RESOLUTION_12B;               /* 12-bit resolution for converted data  */
  hADC.Init.DataAlign             = ADC_DATAALIGN_RIGHT;              /* Right-alignment for converted data */
  hADC.Init.ScanConvMode          = ADC_SCAN_ENABLE;                 /* Scan Mode Disable */
  hADC.Init.EOCSelection          = ADC_EOC_SINGLE_CONV;              /* End of unitary conversion flag */ 
  hADC.Init.LowPowerAutoWait      = DISABLE;                          /* Low power mode disable */
  hADC.Init.ContinuousConvMode    = DISABLE;                          /* Single Conversion  */
  hADC.Init.NbrOfConversion       = 4;                                /* Conversion Number 4 */
  hADC.Init.DiscontinuousConvMode = DISABLE;                          /* Discontinuous Mode Disable */
  /* hADC.Init.NbrOfDiscConversion   = 1; */                          /* Discontinuous Conversion Number 1 */
  hADC.Init.Overrun               = ADC_OVR_DATA_PRESERVED;           /* ADC group regular behavior in case of overrun: data preserved */
  hADC.Init.DMAContinuousRequests = DISABLE;                          /* DMA transfer limited */
  hADC.Init.ExternalTrigConv      = ADC_SOFTWARE_START;               /* Software Trigger */
  hADC.Init.ExternalTrigConvEdge  = ADC_EXTERNALTRIGCONVEDGE_RISING;  /* Rising edge trigger */
  
  if (HAL_ADC_Init(&hADC) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  
  sConfig.Channel      = ADC_CHANNEL_4;             
  sConfig.Rank         = ADC_REGULAR_RANK_1;        
  sConfig.SamplingTime = ADC_SAMPLETIME_640CYCLES_5;           
  if (HAL_ADC_ConfigChannel(&hADC, &sConfig) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  sConfig.Channel      = ADC_CHANNEL_5;             
  sConfig.Rank         = ADC_REGULAR_RANK_2;        
  sConfig.SamplingTime = ADC_SAMPLETIME_640CYCLES_5;           
  if (HAL_ADC_ConfigChannel(&hADC, &sConfig) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  sConfig.Channel      = ADC_CHANNEL_6;             
  sConfig.Rank         = ADC_REGULAR_RANK_3;        
  sConfig.SamplingTime = ADC_SAMPLETIME_640CYCLES_5;           
  if (HAL_ADC_ConfigChannel(&hADC, &sConfig) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  sConfig.Channel      = ADC_CHANNEL_7;             
  sConfig.Rank         = ADC_REGULAR_RANK_4;        
  sConfig.SamplingTime = ADC_SAMPLETIME_640CYCLES_5;           
  if (HAL_ADC_ConfigChannel(&hADC, &sConfig) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  System clock configuration function
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /* Oscillator configuration */
  RCC_OscInitStruct.OscillatorType      = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSI; /* Select oscillator HSI, LSI */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType      |= RCC_OSCILLATORTYPE_HSE; /* Select oscillator HSE */
#endif /* RCC_HSE_SUPPORT */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType      |= RCC_OSCILLATORTYPE_LSE; /* Select oscillator LSE */
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.HSIState            = RCC_HSI_ON;              /* Enable HSI */
  RCC_OscInitStruct.HSIDiv              = RCC_HSI_DIV1;            /* HSI 1 frequency division */
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_8MHz; /* Configure HSI clock 8MHz */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.HSEState            = RCC_HSE_OFF;             /* Close HSE */
  /*RCC_OscInitStruct.HSEFreq             = RCC_HSE_24_32MHz;*/
#endif /* RCC_HSE_SUPPORT */
  RCC_OscInitStruct.LSIState            = RCC_LSI_OFF;             /* Close LSI */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.LSEState            = RCC_LSE_OFF;             /* Close LSE */
  /*RCC_OscInitStruct.LSEDriver           = RCC_LSEDRIVE_MEDIUM;*/
#endif /* RCC_LSE_SUPPORT */
#if defined(RCC_PLL_SUPPORT)
  RCC_OscInitStruct.PLL.PLLState        = RCC_PLL_OFF;             /* Close PLL */
  /*RCC_OscInitStruct.PLL.PLLSource       = RCC_PLLSOURCE_HSI;*/
  /*RCC_OscInitStruct.PLL.PLLMUL          = RCC_PLL_MUL6;*/
  /*RCC_OscInitStruct.PLL.PLLDiv          = RCC_PLL_DIV1;*/
#endif /* RCC_PLL_SUPPORT */
  /* Configure oscillator */
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Clock source configuration */
  RCC_ClkInitStruct.ClockType     = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_PCLK; /* Choose to configure clock HCLK, SYSCLK, PCLK */
  RCC_ClkInitStruct.SYSCLKSource  = RCC_SYSCLKSOURCE_HSISYS; /* Select HSISYS as the system clock */
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;         /* AHB clock 1 division */
  RCC_ClkInitStruct.APBCLKDivider = RCC_HCLK_DIV1;           /* APB clock 1 division */
  /* Configure clock source */
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_0) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
